import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.List;

public class AdbShutterSound {

    private static final List<String> ADB_FILES = Arrays.asList(
        "adb.exe", "AdbWinApi.dll", "AdbWinUsbApi.dll", "etc1tool.exe", "fastboot.exe",
        "hprof-conv.exe", "libwinpthread-1.dll", "make_f2fs_casefold.exe", "make_f2fs.exe",
        "mke2fs.conf", "mke2fs.exe", "NOTICE.txt", "source.properties", "sqlite3.exe"
    );

    public static void main(String[] args) {
        Path tempDir = null;
        try {
            tempDir = unpackAdb();
            Path adbExecutable = tempDir.resolve("adb.exe");
            
            System.out.println("Checking for ADB devices...");
            boolean deviceAuthorized = false;

            // Wait for up to 60 seconds (12 retries * 5 seconds)
            for (int i = 0; i < 12; i++) {
                CommandResult adbDevicesResult = executeCommand(adbExecutable.toString(), "devices");
                String adbDevicesOutput = adbDevicesResult.stdout;

                if (adbDevicesOutput.matches("(?s).*List of devices attached.*\\sdevice\\s*")) {
                    System.out.println("Device is authorized.");
                    deviceAuthorized = true;
                    break; // Exit loop
                } else if (adbDevicesOutput.contains("unauthorized")) {
                    System.out.println("Device is unauthorized. Please check your phone to allow the RSA key fingerprint.");
                    System.out.println("Waiting... Retrying in 5 seconds. (" + (i + 1) + "/12)");
                } else {
                    System.out.println("No authorized device found. Please connect your device and enable USB debugging.");
                    System.out.println("Waiting... Retrying in 5 seconds. (" + (i + 1) + "/12)");
                }
                
                Thread.sleep(5000); // Wait 5 seconds
            }

            if (deviceAuthorized) {
                System.out.println("Proceeding to check shutter sound setting...");
                CommandResult getSoundSettingResult = executeCommand(adbExecutable.toString(), "shell", "settings", "get", "system", "csc_pref_camera_forced_shuttersound_key");
                String currentSoundSetting = getSoundSettingResult.stdout.trim();

                if ("1".equals(currentSoundSetting)) {
                    System.out.println("Camera shutter sound is currently enabled (value: 1). Disabling it...");
                    executeCommand(adbExecutable.toString(), "shell", "settings", "put", "system", "csc_pref_camera_forced_shuttersound_key", "0");
                    
                    CommandResult verifySoundSettingResult = executeCommand(adbExecutable.toString(), "shell", "settings", "get", "system", "csc_pref_camera_forced_shuttersound_key");
                    String newSoundSetting = verifySoundSettingResult.stdout.trim();

                    if ("0".equals(newSoundSetting)) {
                        System.out.println("Camera shutter sound disabled successfully.");
                    } else {
                        System.out.println("Failed to disable camera shutter sound. The current value is: " + newSoundSetting);
                    }
                } else if ("0".equals(currentSoundSetting)) {
                    System.out.println("Camera shutter sound is already disabled.");
                } else {
                     System.out.println("Could not determine shutter sound status. Current value: '" + currentSoundSetting + "'");
                }
            } else {
                System.out.println("\nTimeout reached. No authorized device found. Exiting.");
            }

        } catch (IOException | InterruptedException e) {
            System.out.println("An error occurred: " + e.getMessage());
            e.printStackTrace();
        } finally {
            System.out.println("\nScript execution completed at " + new java.util.Date());
            System.out.println("=========================================");
            System.out.println("Press any key to exit...");
            try {
                System.in.read();
            } catch (IOException e) {
                // Ignore
            }
        }
    }

    private static Path unpackAdb() throws IOException {
        Path tempDir = Files.createTempDirectory("adb-temp-");
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            try {
                Files.walk(tempDir).sorted((a, b) -> b.compareTo(a)).forEach(path -> {
                    try { Files.delete(path); } catch (IOException e) { /* ignore */ }
                });
            } catch (IOException e) { /* ignore */ }
        }));
        for (String fileName : ADB_FILES) {
            try (InputStream is = AdbShutterSound.class.getResourceAsStream("/adb/" + fileName)) {
                if (is == null) throw new IOException("Cannot find resource: /adb/" + fileName);
                Files.copy(is, tempDir.resolve(fileName), StandardCopyOption.REPLACE_EXISTING);
            }
        }
        return tempDir;
    }

    private static CommandResult executeCommand(String... command) throws IOException, InterruptedException {
        ProcessBuilder pb = new ProcessBuilder(command);
        Process process = pb.start();
        StringBuilder stdout = new StringBuilder();
        try (BufferedReader stdInput = new BufferedReader(new InputStreamReader(process.getInputStream()))) {
            String s;
            while ((s = stdInput.readLine()) != null) {
                stdout.append(s).append(System.lineSeparator());
            }
        }
        // Reading stderr is good practice but omitted for brevity here as stdout has enough info for this app
        int exitCode = process.waitFor();
        return new CommandResult(exitCode, stdout.toString(), ""); // Simplified
    }

    private static class CommandResult {
        final int exitCode;
        final String stdout;
        final String stderr;
        CommandResult(int exitCode, String stdout, String stderr) {
            this.exitCode = exitCode; this.stdout = stdout; this.stderr = stderr;
        }
    }
}