// 스타일 주입
const style = document.createElement('style');
style.textContent = `
  .highlight-name {
    background-color: #E61E2B !important;
    color: white !important;
    font-weight: bold !important;
  }
  
  .alert-box {
    position: fixed;
    top: 20px;
    right: 20px;
    background-color: #E61E2B;
    color: white;
    padding: 10px;
    border-radius: 5px;
    z-index: 999999;
    display: flex;
    align-items: center;
    gap: 10px;
    font-size:1.2em;
  }
  
  .close-button {
    width: 20px;
    height: 20px;
    border: 1px solid white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    background: none;
    color: white;
    font-size: 12px;
    padding: 0;
    margin: 0;
  }

  .close-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
  }
`;
document.head.appendChild(style);

// 정치 키워드 체크
function isPoliticalContent(text) {
    const keywords = ['정부', '국회', '의회', '정당', '여당', '야당', 
                     '보수', '진보', '정책', '선거', '투표', '국민의힘', '국민의 힘', '민주당', '탄핵',
                     '내란', '대통령', '총선', '반란'];
    let matches = 0;
    keywords.forEach(keyword => {
        if (text.includes(keyword)) matches++;
    });
    return matches >= 2;
}

// 이름 하이라이트 처리
function highlightNames(names) {
    // 텍스트 노드만 선택
    const textNodes = [];
    const walk = document.createTreeWalker(
        document.body,
        NodeFilter.SHOW_TEXT,
        {
            acceptNode: function(node) {
                // script나 style 태그는 제외
                const parent = node.parentElement;
                if (!parent || ['SCRIPT', 'STYLE', 'NOSCRIPT'].includes(parent.tagName)) {
                    return NodeFilter.FILTER_REJECT;
                }
                // 빈 텍스트는 제외
                if (!node.textContent.trim()) {
                    return NodeFilter.FILTER_REJECT;
                }
                return NodeFilter.FILTER_ACCEPT;
            }
        }
    );

    while (walk.nextNode()) {
        textNodes.push(walk.currentNode);
    }

    // 각 텍스트 노드에 대해 처리
    textNodes.forEach(node => {
        let text = node.textContent;
        let hasMatch = false;
        
        names.forEach(name => {
            if (name && text.includes(name)) {
                hasMatch = true;
                // span으로 감싸기
                text = text.replace(new RegExp(name, 'g'), 
                    `<span class="highlight-name">${name}</span>`);
            }
        });
        
        if (hasMatch && node.parentElement) {
            // wrapper element 생성
            const wrapper = document.createElement('span');
            wrapper.innerHTML = text;
            
            try {
                node.parentElement.replaceChild(wrapper, node);
            } catch (error) {
                console.error('Error replacing node:', error);
            }
        }
    });
}

// 알림창 표시
function showAlert() {
    const alert = document.createElement('div');
    alert.className = 'alert-box';
    
    // 메시지 텍스트
    const messageSpan = document.createElement('span');
    messageSpan.textContent = '12.3 비상계엄 관련인이 감지되었습니다!';
    
    // 닫기 버튼
    const closeButton = document.createElement('button');
    closeButton.className = 'close-button';
    closeButton.textContent = 'X';
    closeButton.onclick = function() {
        alert.remove();
    };
    
    // 요소들 추가
    alert.appendChild(messageSpan);
    alert.appendChild(closeButton);
    
    document.body.appendChild(alert);
    
    // 10초 후 자동 제거
    setTimeout(() => {
        if (alert.parentElement) alert.remove();
    }, 10000);
}

// 메인 함수
async function main() {
    try {
        // 페이지가 정치 콘텐츠인지 확인
        if (!isPoliticalContent(document.body.textContent)) return;
        
        // list.txt 읽기
        const response = await fetch(chrome.runtime.getURL('list.txt'));
        const text = await response.text();
        const names = text.split('\n')
            .map(name => name.trim())
            .filter(name => name.length > 0);
        
        if (names.length === 0) return;
        
        // 텍스트에서 이름 찾기
        let foundAnyName = false;
        const pageText = document.body.textContent;
        for (const name of names) {
            if (pageText.includes(name)) {
                foundAnyName = true;
                break;
            }
        }
        
        // 이름이 발견된 경우에만 하이라이트와 알림 표시
        if (foundAnyName) {
            highlightNames(names);
            showAlert();
        }
        
    } catch (error) {
        console.error('Error:', error);
    }
}

// 실행
setTimeout(main, 1000); // 페이지 로드 후 1초 뒤에 실행